%**************************************************************************
% Function Name  : VN_Serial 
% Description    : 
%  Opens a new serial port for communication with a VectorNav sensor. 
%  The function will automatically open the given COM port. It will then
%  scan thru each possible baud rate and reset the chip to 9600 baud. 
%  At this point it will then connect either at the user defined baud rate,
%  or if no baud rate is provided it will connect at the highest baud rate
%  that is supported by the machine for the given COM port.  In most cases
%  you will be able to connect at higher baud rates using the USB
%  connection as opposed to the RS-232 serial port. For best results you
%  will want to ensure that the latency in windows is set to 1ms.
%
% Input(s)       : comPort  -> COM port name or number ('COM8' or 8)
%                : BaudRate -> optional baud rate
% Output(s)      : s        -> serial port object
%**************************************************************************
% Examples:
%
%   s = VNserial('COM8');
%   VNwriteRegister(s, 23, [1 0 0 0 1 0 0 0 1 0 0 0]);
%   fclose(s);
%
% ------------- VectorNav Technologies, LLC -------------------------------
% This file is property of VectorNav Technologies and cannot be used,
% copied or distributed without the written consent of VectorNav
% Technologies. 
% -------------------------------------------------------------------------
function s = VNserial(comPort, BaudRate)

%Convert input to string
if isstr(comPort) == 0
    comPort = ['COM' num2str(comPort)];
end

%Check to see if COM port is already open, if so then close COM port.
ports = instrfind;
for i=1:length(ports)
    if strcmp(ports(i).Port, comPort) == 1
        fclose(ports(i));
    end
end

%Baud rates
Brates = [4800 9600 19200 38400 57600 115200 230400 460800 921600];

%Create the serial port
s = serial(comPort, 'BaudRate', 9600);

%Create the serial port
fopen(s);
for i=1:length(Brates)
    try
        s.BaudRate = Brates(i);
        VNprintf(s, 'VNWRG,5,9600');
    catch
        Bmax = Brates(i-1);
        break
    end
    Bmax = Brates(i);
end

%Connect at 9600
s.BaudRate = 9600;
VNprintf(s, 'VNWRG,6,0');
VNprintf(s, 'VNWRG,7,200');
pause(0.1);
VNclearbuffer(s);

%Connect at user selected baud rate or highest found without error
if nargin < 2
    BaudRate = Bmax;
end

if(BaudRate > Bmax)
    fprintf('\nUnable to connect at desired baud rate of %i.\nThe baud rate has been set to the maximum allowable for this COM port.\n\n', BaudRate);
    BaudRate = Bmax;
end

%Connect at given baud rate
VNsetbaudrate(s, BaudRate);


