﻿//*****************************************************************************************
//                           LICENSE INFORMATION
//*****************************************************************************************
//   Fieldforce TCM-Modul Version 1.0.0.0
//
//   Class file for managing serial port communication to TCM
//
//   Copyright (C) 2012  
//   Jürgen Goldmann
//   Email: j.goldmann@dlg.org
//   
//
//   This program is free software: you can redistribute it and/or modify
//   it under the terms of the GNU General Public License as published by
//   the Free Software Foundation, either version 3 of the License, or
//   (at your option) any later version.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program.  If not, see <http://www.gnu.org/licenses/>.
//*****************************************************************************************
using System;
using System.Collections.Generic;
using System.Text;
using System.Windows.Forms;

namespace PCComm
{
    //Expample for TCM communication
    //More Infos seee TCM User-Manual

    //00 15  [Length in byte]
    //05 [kDataResp]
    //03 [Count of Datavalues]
    //05 [05d Heading] 8B 79 7A 43 [Value]  --> compared to TCM-Programm: 250.6
    //18 [24d Pitch  ] 38 39 37 3F [Value]  --> compared to TCM-Programm: 0.6
    //19 [25d Roll   ] D4 13 96 3F [Value]  --> compared to TCM-Programm: 1.1
    //D0 FB [Checksum]


    class TCMprotocol
    {
        public Dictionary<int, float> FloatValue = new Dictionary<int, float>();
        //List<float> FloatList = new List<float>();
        byte [] _ba;
        int _iByteCount = 0;
        int _iIDCount = 0;
        int _iByteDefLength = 0;
        int _iCommandID = 0;
        bool bDoInit = true;

        private RichTextBox _rtb;
        private TextBox _textbox1, _textbox2, _textbox3;
        private TrackBar _tb1, _tb2;


        public void SetTextBoxes(TextBox t1, TextBox t2, TextBox t3)
        {
            _textbox1 = t1;
            _textbox2 = t2;
            _textbox3 = t3;

        }

        public void SetTrackBar(TrackBar tRoll, TrackBar tPitch)
        {
            _tb1 = tRoll;
            _tb2 = tPitch;
        }

        public void SetRichTextbox(RichTextBox rtb)
        {
            _rtb = rtb;
        }

        /// <summary>
        /// constructor
        /// </summary>
        /// 
        public TCMprotocol()
        {
            Init(); 
        }

        /// <summary>
        /// Init communication
        /// </summary>
        public void Init()
        {
           
            _iByteCount = 0;
            _iIDCount = 0;
            _iByteDefLength = 0;
            _iCommandID = 0;
            _ba = new byte[255];
            _ba.Initialize();
        }

        /// <summary>
        /// Read data from TCM
        /// </summary>
        /// <param name="b"></param>
        public void ReadValues(byte[] b)
        {
            if (bDoInit)
            {
                bDoInit = false;
                Init();
            }

            if (_iByteCount > 255)
            {
                bDoInit = true;
            }

            if (_iByteCount <= 3)
            {
                // runs until now only with messages < 255 byte !!!
                // so the first byte "0" could used as start-byte !!!
                try
                {
                    for (int i = 0; i < b.Length; i++)
                    {
                        _ba[_iByteCount] = b[i];
                        if (_ba[0] > 0)
                        {
                            bDoInit = true;
                        }

                        if (_iByteCount == 1) // Length
                        {
                            _iByteDefLength = _ba[1];
                        }
                        if (_iByteCount == 2) // CommandID
                        {
                            _iCommandID = _ba[2];
                        }
                        if (_iByteCount == 3) // ID-Count
                        {
                            _iIDCount = _ba[3];
                        }
                        _iByteCount++;
                    }
                }
                catch { bDoInit = true; }

            }
            else
            {
                for (int i = 0; i < b.Length; i++)
                {
                    try
                    {
                        _ba[_iByteCount] = b[i];
                        _iByteCount++;
                    }
                    catch  { bDoInit = true; }
                }

                
                if (_iByteCount >= _iByteDefLength)
                {

                    // CRC calculation until now not implementation
                    // USER CRC class
                                       

                    // value calculation
                    if (_ba[2] == 05) // CommandID: kDataResp
                    {
                        int iOffset = 4;

                        DisplayData(_rtb, DateTime.Now.ToString("hh:mm:ss.fff")+ "; ");

                        for (int i = 0; i < _iIDCount; i++)
                        {
                            int DataID = _ba[iOffset];
                            iOffset++;
                            switch (DataID)
                            {
                                case  5: // kHeading
                                case 24: // kPAngle (=pitch)
                                case 25: // pRAngle (=roll)
                                    byte[] v = new byte[4];
                                    for (int ii = 0; ii < 4; ii++)
                                    {
                                        v[ii] = _ba[iOffset];
                                        iOffset++;
                                    }
                                    FloatValue[DataID] = TCMprotocol.ByteArrayToFloat32(v);
                                   
                                   DisplayData(_rtb,  FloatValue[DataID].ToString("000.000")+ "; ");
                                   
                                    switch (DataID)
                                    {
                                        case 5:
                                            DisplayValue(_textbox1, FloatValue[DataID].ToString());
                                            break;
                                        case 24:
                                            DisplayValue(_textbox2, FloatValue[DataID].ToString());
                                            DisplaySlider(_tb1, (int)FloatValue[DataID] + 90);
                                            break;
                                        case 25:
                                            DisplayValue(_textbox3, FloatValue[DataID].ToString());
                                            DisplaySlider(_tb2, (int)FloatValue[DataID] + 90);
                                            break;
                                    }
                                    break;

                                default:
                                    break;
                            }
                        } //for end
                        DisplayData(_rtb, "\n");
                    }
                    else
                    {
                        // not implemented
                    }

                    _iByteCount = 0;
                    _iByteDefLength = 0;
                }
            }

        }


        /// <summary>
        /// 
        /// </summary>
        /// <param name="tb"></param>
        /// <param name="msg"></param>
        [STAThread]
        private void DisplayValue(TextBox tb, string msg)
        {
           tb.Invoke(new EventHandler
                         (delegate  {tb.Text = msg;})
                     );
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="tb"></param>
        /// <param name="val"></param>
        private void DisplaySlider (TrackBar tb, Int32 val)
        {
            tb.Invoke(new EventHandler(delegate { try { tb.Value = val; } catch { } }));
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="tb"></param>
        /// <param name="msg"></param>
        private void DisplayData(RichTextBox tb, string msg)
        {
            
            tb.Invoke(new EventHandler(
            delegate { 
                      try 
                      {
                          tb.SelectedText = string.Empty;
                          tb.AppendText(msg);
                          tb.ScrollToCaret();
                      } 
                      catch { } 
                    }
                                      )
                    );
        }



        // Static helper procedures ----------------------------------------

        /// <summary>
        /// create TCM datagram
        /// Byte-Length (at begin) and CRC (at end) automatic added to protocoll
        /// </summary>
        /// <param name="bain"></param>
        /// <returns></returns>
        public static byte[] CreateTCMDatagram(byte[] bain)
        {
            // length calc.
            byte[] bacrc = new byte[bain.Length + 2];
            bacrc[0] = 0;
            bacrc[1] = (byte)(bain.Length + 4);
            Array.Copy(bain, 0, bacrc, 2, bain.Length);
            // CRC calc.
            CRC16 cr = new CRC16(0);
            byte[] bcc = cr.ComputeHighLowByte(bacrc);

            byte[] ba = new byte[bacrc.Length + 2];
            Array.Copy(bacrc, ba, bacrc.Length);
            ba[ba.Length - 2] = bcc[0];
            ba[ba.Length - 1] = bcc[1];

            return ba;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="ba"></param>
        /// <returns></returns>
        public static float ByteArrayToFloat32(byte[] ba)
        {
            //Array.Reverse(ba);

            Single x = BitConverter.ToSingle(ba, 0);
            return x;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="value"></param>
        /// <returns></returns>
        public static byte[] UInt32ToByteArray(UInt32 value)
        {
            byte[] x = BitConverter.GetBytes(value);
            return x;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="value"></param>
        /// <returns></returns>
        public static byte[] Float32ToByteArray(System.Single value)
        {
            byte[] x = BitConverter.GetBytes(value);
            return x;
        }

        /// <summary>
        /// convert Byte to Bool
        /// </summary>
        /// <param name="b"></param>
        /// <returns></returns>
        public static bool ByteToBool(byte b)
        {
            if (b == 0)
                return false;
            else
                return true;
        }

        
        
        /// <summary>
        /// reverse byte order (16-bit)
        /// </summary>
        /// <param name="value"></param>
        /// <returns></returns>
        public static UInt16 ReverseBytes(UInt16 value)
        {
            return (UInt16)((value & 0xFFU) << 8 | (value & 0xFF00U) >> 8);
        }

        // reverse byte order (32-bit)
        public static UInt32 ReverseBytes(UInt32 value)
        {
            return (value & 0x000000FFU) << 24 | (value & 0x0000FF00U) << 8 |
             (value & 0x00FF0000U) >> 8 | (value & 0xFF000000U) >> 24;
        }



        // reverse byte order (64-bit)
        public static UInt64 ReverseBytes(UInt64 value)
        {
            return (value & 0x00000000000000FFUL) << 56 | (value & 0x000000000000FF00UL) << 40 |
                   (value & 0x0000000000FF0000UL) << 24 | (value & 0x00000000FF000000UL) << 8 |
                   (value & 0x000000FF00000000UL) >> 8 | (value & 0x0000FF0000000000UL) >> 24 |
                   (value & 0x00FF000000000000UL) >> 40 | (value & 0xFF00000000000000UL) >> 56;
        }


    }
}
